<?php
/* --------------------------------------------------------------
   LanguageFilter.php 2020-04-16
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Core\Language\SqlCriteria;

use Gambio\Core\Criteria\SqlFilter;
use Webmozart\Assert\Assert;

/**
 * Class LanguageFilter
 *
 * @package Gambio\Core\Language\SqlCriteria
 */
class LanguageFilter implements SqlFilter
{
    /**
     * @var string
     */
    private $attribute;
    
    /**
     * @var string
     */
    private $value;
    
    /**
     * @var string
     */
    private $operation;
    
    /**
     * @var string[]
     */
    private $attributeColumnMapping = [
        'id'          => 'languages_id',
        'name'        => 'name',
        'code'        => 'code',
        'status'      => 'status',
        'statusAdmin' => 'status_admin',
    ];
    
    /**
     * @var string[]
     */
    private $allowedForLikeOperation = [
        'name',
        'code',
    ];
    
    /**
     * @var string[]
     */
    private $allowedForNumericOperation = [
        'id',
    ];
    
    
    /**
     * LanguageFilter constructor.
     *
     * @param string $attribute
     * @param string $value
     * @param string $operation
     */
    private function __construct(string $attribute, string $value, string $operation = 'like')
    {
        $attributes = array_keys($this->attributeColumnMapping);
        Assert::oneOf($attribute,
                      $attributes,
                      'Provided attribute needs to be one of: ' . implode(', ', $attributes) . '; Got: ' . $attribute);
        
        $operations = ['like', 'eq', 'gt', 'gte', 'lt', 'lte'];
        Assert::oneOf($operation,
                      $operations,
                      'Provided operation needs to be one of: ' . implode(', ', $operations) . '; Got: ' . $operation);
        
        if ($operation === 'like') {
            Assert::oneOf($attribute,
                          $this->allowedForLikeOperation,
                          'Provided attribute is not allowed for "like" operation.');
        } elseif (in_array($operation, ['gt', 'gte', 'lt', 'lte'])) {
            Assert::oneOf($attribute,
                          $this->allowedForNumericOperation,
                          'Provided attribute is not allowed for a numeric operation.');
        }
        
        $this->attribute = $attribute;
        $this->value     = $value;
        $this->operation = $operation;
    }
    
    
    /**
     * @param string $attribute
     * @param string $value
     * @param string $operation
     *
     * @return LanguageFilter
     */
    public static function create(string $attribute, string $value, string $operation = 'like'): LanguageFilter
    {
        return new self($attribute, $value, $operation);
    }
    
    
    /**
     * @return string
     */
    public function attribute(): string
    {
        return $this->attribute;
    }
    
    
    /**
     * @return string
     */
    public function value(): string
    {
        return $this->value;
    }
    
    
    /**
     * @return string
     */
    public function operation(): string
    {
        return $this->operation;
    }
    
    
    /**
     * @return string
     */
    public function column(): string
    {
        return $this->attributeColumnMapping[$this->attribute];
    }
}